/************************************************************************
 * \file: prxy_io.c
 *
 * \version: $Id: trace_proxy_io.c,v 1.6 2012/05/30 08:25:18 jayanth.mc Exp $
 *
 * This file implements the IO handling for Proxy.
 *
 * \component: Gen2 Trace
 *
 * \author B. Das         bhaskar.das(o)in.bosch.com
 *         Sakthivelu S.  sakthivelu.s(o)in.bosch.com
 * \modified: Arun V
 * \copyright: (c) 2003 - 2009 ADIT
 *
 ***********************************************************************/

#include "trace_base.h"
#include <pfcfg_interface.h>
#include <sys/prctl.h>



/**
 * Search for Trace proxy binary file for overwrite
 *
 * \parm mgr  Pointer to Trace manager.
 * \parm idx  Pointer to variable holding prev_count.
 *
 * return None.
 */
EXPORT void TRACE_chk_prev_bin_file(TRACE_mgr* mgr, U32* idx)
{
  U8 bfec[sizeof(U32)] = {0};
  TRACE_PROXY_HDR hdr ;
  U32            cnt   = (*idx)-1;
  ssize_t         sz   = 0;
  S32             fd   = -1;

  mgr = mgr; 
  memset(&hdr, 0, sizeof(TRACE_PROXY_HDR));
  
  sprintf((VP)bfec, "%u", cnt);
  strcpy((VP)mgr->prxy.file_nm, (VP)mgr->sh->prxy.bin_file_path);
  strcat((VP)mgr->prxy.file_nm, (VP)TRACE_PROXY_BIN_FILE_NM);
  strcat((VP)mgr->prxy.file_nm, (VP)bfec);
  strcat((VP)mgr->prxy.file_nm, (VP)TRACE_PROXY_BIN_FILE_EXTN);

  fd = open((VP)mgr->prxy.file_nm, TRACE_FILE_R_PLUS, TRACE_FILE_ACCESS_MODE);
  if(fd >= E_OK)
  {
    /* Validate bin file */
    sz = read(fd, (VP)&hdr, sizeof(hdr));
    if(((size_t)sz == sizeof(hdr)) && (hdr.sign == TRACE_PROXY_FILE_SIGN))
    {
      if(hdr.ms.mode_stat.file_stat != TRACE_BIN_FILE_FULL)
      {
        *idx = cnt;
      }
    }
    (void)close(fd);
  }
}

/**
 * Create Trace proxy binary file.
 *
 * \parm cnt  Pointer to variable holding prev_count.
 * \parm mgr  Pointer to Trace manager.
 *
 * return None.
 */
EXPORT void TRACE_cre_bin_file_cnt(U32* prev_cnt, TRACE_mgr* mgr)
{
  U8 bfec[sizeof(U32)] = {0};
  S32             fd  = -1;
  ssize_t         asz = 0;
  U32             sz  = 0;
  S32             written_bytes = 0;

  strcpy((VP)mgr->prxy.file_nm, (VP)mgr->sh->prxy.bin_file_path);
  strcat((VP)mgr->prxy.file_nm, (VP)TRACE_PROXY_DAT_FILE_NM);
  fd = open((VP)mgr->prxy.file_nm, TRACE_FILE_R_PLUS, TRACE_FILE_ACCESS_MODE);

  if(fd < E_OK)
  {
    fd = open((VP)mgr->prxy.file_nm, TRACE_FILE_W_PLUS, TRACE_FILE_ACCESS_MODE);
  }

  if(mgr->sh->prxy.lookup_filecnt == TRUE)
  {
    mgr->sh->prxy.cur_bfile_cnt = 1;
    /* lookup .dat file for running file cnt */
    if(fd >= E_OK)
    {
      asz = read(fd, (VP)&sz, sizeof(mgr->sh->prxy.cur_bfile_cnt));
      if(asz == (ssize_t)sizeof(mgr->sh->prxy.cur_bfile_cnt))
      {
        if(sz <= mgr->sh->prxy.max_bfile_cnt)
        {
          if(mgr->sh->prxy.flush_mode == FALSE)
          {
            TRACE_chk_prev_bin_file(mgr, &sz);
          }
          mgr->sh->prxy.cur_bfile_cnt = sz;
        }
      }
      (void)lseek(fd, 0, SEEK_SET);
    }
    mgr->sh->prxy.lookup_filecnt = FALSE;
  }
  sprintf((VP)bfec, "%u", mgr->sh->prxy.cur_bfile_cnt);

  strcpy((VP)mgr->prxy.file_nm, (VP)mgr->sh->prxy.bin_file_path);
  strcat((VP)mgr->prxy.file_nm, (VP)TRACE_PROXY_BIN_FILE_NM);
  strcat((VP)mgr->prxy.file_nm, (VP)bfec);
  strcat((VP)mgr->prxy.file_nm, (VP)TRACE_PROXY_BIN_FILE_EXTN);

  /* update running file count */
  if(fd >= E_OK)
  {
    sz = mgr->sh->prxy.cur_bfile_cnt + 1;
    written_bytes = write(fd, (VP)&sz, sizeof(mgr->sh->prxy.cur_bfile_cnt));
    if(0 == written_bytes)
    {
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Bytes written is Zero\n");
    }
    (void)fsync(fd);
    (void)close(fd);
  }
  (*prev_cnt) = mgr->sh->prxy.cur_bfile_cnt;
}

LOCAL ER TRACE_flush_wr(TRACE_mgr* mgr, U8* pkt, U32 size);
LOCAL ER TRACE_flush_wr(TRACE_mgr* mgr, U8* pkt, U32 size)
{
  S32             fd          = -1;
  ssize_t          sz         = 0;
  ER              rc          = E_FAIL;
  S32            mode         = TRACE_FILE_R_PLUS;
  BOOL           bCreateNew   = FALSE;
  TRACE_PROXY_HDR hdr         ;
  TRACE_PROXY_HDR default_hdr ;
  static U32       file_cnt    = 1;
  TRACE_PRXY_MSG  msg         ;
  U8*             buf         = (VP)&msg.msg_ptn_2;

  memset(&hdr, 0, sizeof(TRACE_PROXY_HDR));
  default_hdr = hdr;
  memset(&msg, 0, sizeof(TRACE_PRXY_MSG));
  
  msg.msg_ptn_1 = TRACE_PROXY_MSG_SIGN;
  msg.msg_ptn_2 = TRACE_PROXY_MSG_SIGN; 
  default_hdr.sign                   = TRACE_PROXY_FILE_SIGN;
  default_hdr.ms.mode_stat.file_mode = mgr->sh->prxy.flush_mode;
  default_hdr.ms.mode_stat.file_stat = TRACE_BIN_FILE_EMPTY;
  default_hdr.over_wr_cnt            = 0;

  while((TRUE != rc) && (FALSE == mgr->prxy.exit))
  {
    rc = E_FAIL;
    if(file_cnt != mgr->sh->prxy.cur_bfile_cnt)
    {
      TRACE_cre_bin_file_cnt(&file_cnt, mgr);
      bCreateNew = TRUE;
    }

    /* Open bin file */
    mode = (bCreateNew == TRUE)? TRACE_FILE_W_PLUS :
                                  TRACE_FILE_R_PLUS;
    fd = open((VP)mgr->prxy.file_nm, mode, TRACE_FILE_ACCESS_MODE);
    if(fd >= E_OK)
    {
      if(bCreateNew == TRUE)
      {
        /* write bin file header */
        sz = write(fd, &default_hdr, sizeof(default_hdr));
        if((size_t)sz != sizeof(default_hdr))
        {
          break;
        }
        bCreateNew = FALSE;
      }
      else
      {
        /* Validate bin file */
        sz = read(fd, &hdr, sizeof(hdr));
        if(((size_t)sz != sizeof(hdr)) || (hdr.sign != TRACE_PROXY_FILE_SIGN))
        {
          break;
        }
      }
    }
    if(fd >= E_OK)
    {
      sz = TRACE_get_file_sz(fd);
      if(mgr->sh->prxy.timestamp_mode == TRUE)
      {
        size += (sizeof(msg.msg_ptn_1));
/*This work around for truncation of trace messages while time stamp is enabled needs to be revisited As per current protocol, only one byte frame count is present*/
        /*size += (sizeof(msg.msg_ptn_1)- TRACE_MIN_FRAME_CNT );*/
      }
      size += (sizeof(msg.msg_ptn_2)- TRACE_MIN_FRAME_CNT );

      if(((size + (size_t)sz) <= mgr->sh->prxy.bfile_sz))
      {
        if(mgr->sh->prxy.timestamp_mode == TRUE)
        {
          (void)memcpy((VP)&msg.tm_msg,&pkt[1],pkt[1]);
          /* update size as per timestamp format */
          size += (sizeof(msg.tm_msg)- pkt[1]);
	  pkt += (pkt[1]);
/*pkt[1] contains the length of the message buffer, instead of (pkt[1]+1)*/
          /*pkt += (pkt[1]+ 1);*/
          buf = (VP)&msg.msg_ptn_1;
        }
        (void)memcpy((VP)&msg.trc_msg, &pkt[1],pkt[1]);

        /* Write trace packet */
        (void)lseek(fd, 0, SEEK_END);
        sz = write(fd, buf, size);
        if((size_t)sz == size)
        {
          rc = TRUE;
        }
      }
      else
      {
       /* update file state */
       hdr.ms.mode_stat.file_stat = TRACE_BIN_FILE_FULL;
       (void)lseek(fd,0, SEEK_SET);
       sz = write(fd, (VP)&hdr, sizeof(hdr));
       if((size_t)sz != sizeof(hdr))
       {
          break;
       }
       (void)fsync(fd);
       (void)close(fd);
       fd = -1;

       mgr->sh->prxy.cur_bfile_cnt++;
       if(mgr->sh->prxy.cur_bfile_cnt > mgr->sh->prxy.max_bfile_cnt)
       {
         /* reset to 1 */
         mgr->sh->prxy.cur_bfile_cnt = 1;
       }
      }
    }
    if((TRUE == rc) && (fd >= E_OK))
    {
      (void)fsync(fd);
      (void)close(fd);
      fd = -1;
    }
  }
  if((E_FAIL == rc) && (fd >= E_OK))
  {
    (void)fsync(fd);
    (void)close(fd);
  }

  return rc;
}

LOCAL ER TRACE_ovr_wr(TRACE_mgr* mgr, U8* pkt, U32 size);
LOCAL ER TRACE_ovr_wr(TRACE_mgr* mgr, U8* pkt, U32 size)
{
  S32             fd          = -1;
  ssize_t          sz         = 0;
  ER              rc          = E_FAIL;
  BOOL           bNewFile     = FALSE;
  TRACE_PROXY_HDR hdr         ;
  TRACE_PROXY_HDR default_hdr ;
  static U32       file_cnt    = 1;
  TRACE_PRXY_MSG  msg         ;
  U8*             buf         = (VP)&msg.msg_ptn_2;
  U8 dummy_buf[TR_PROXY_MAX_MSG_SZ] = {0};
  S32             written_bytes = 0;

  memset(&hdr, 0, sizeof(TRACE_PROXY_HDR));
  default_hdr = hdr;
  memset(&msg, 0, sizeof(TRACE_PRXY_MSG));

  msg.msg_ptn_1 = TRACE_PROXY_MSG_SIGN; 
  msg.msg_ptn_2 = TRACE_PROXY_MSG_SIGN; 
  default_hdr.sign                   = TRACE_PROXY_FILE_SIGN;
  default_hdr.ms.mode_stat.file_mode = mgr->sh->prxy.flush_mode;
  default_hdr.ms.mode_stat.file_stat = TRACE_BIN_FILE_EMPTY;
  default_hdr.over_wr_cnt            = 0;

  while((TRUE != rc) && (FALSE == mgr->prxy.exit))
  {
    rc = E_FAIL;
    if(file_cnt != mgr->sh->prxy.cur_bfile_cnt)
    {
      TRACE_cre_bin_file_cnt(&file_cnt, mgr);
      bNewFile = TRUE;
    }

    /* Open bin file */
    fd = open((VP)mgr->prxy.file_nm, TRACE_FILE_R_PLUS, TRACE_FILE_ACCESS_MODE);
    if(fd >= E_OK)
    {
      /* Validate bin file */
      sz = read(fd, &hdr, sizeof(hdr));
      if(((size_t)sz != sizeof(hdr)) || (hdr.sign != TRACE_PROXY_FILE_SIGN))
      {
        break;
      }
      TRACE_pack_32bit_in_bigendian(mgr->sh->prxy.cur_sz, (VP)&hdr.over_wr_cnt);

      if(bNewFile == TRUE)
      {
        /* Is file full? */
        if(hdr.ms.mode_stat.file_stat == TRACE_BIN_FILE_FULL)
        {
          mgr->sh->prxy.cur_sz = sizeof(hdr);
          hdr.ms.mode_stat.file_stat = TRACE_BIN_FILE_PARTLY_FULL;
          TRACE_pack_32bit_in_bigendian(mgr->sh->prxy.cur_sz,(VP)&hdr.over_wr_cnt);
          (void)lseek(fd, 0, SEEK_SET);
          sz = write(fd, &hdr, sizeof(hdr));
          if((size_t)sz != sizeof(hdr))
          {
            break;
          }
        }
        /* Is file not full? */
        if(hdr.ms.mode_stat.file_stat == TRACE_BIN_FILE_EMPTY)
        {
          mgr->sh->prxy.cur_sz = (size_t)TRACE_get_file_sz(fd);
        }
        /* Is file Part? */
        if(hdr.ms.mode_stat.file_stat == TRACE_BIN_FILE_PARTLY_FULL)
        {
          TRACE_get_32bit_frm_bigendian((VP)&hdr.over_wr_cnt,
                                    (VP)&mgr->sh->prxy.cur_sz);
        }
        bNewFile = FALSE;
      }
    }
    else
    {
      /* file not present. create newly */
      fd = open((VP)mgr->prxy.file_nm, TRACE_FILE_W_PLUS, TRACE_FILE_ACCESS_MODE);
      if(fd >= E_OK)
      {
        /* write bin file header */
        sz = write(fd, &default_hdr, sizeof(default_hdr));
        if((size_t)sz != sizeof(default_hdr))
        {
          break;
        }
        mgr->sh->prxy.cur_sz = sizeof(default_hdr);
        hdr = default_hdr;
      }
    }

    if(fd >= E_OK)
    {
      sz = (ssize_t)mgr->sh->prxy.cur_sz;
      if(mgr->sh->prxy.timestamp_mode == TRUE)
      {
        size += (sizeof(msg.msg_ptn_1)- TRACE_MIN_FRAME_CNT );
      }
      size += (sizeof(msg.msg_ptn_2)- TRACE_MIN_FRAME_CNT );

      if(((size + (size_t)sz) <= mgr->sh->prxy.bfile_sz))
      {
        if(mgr->sh->prxy.timestamp_mode == TRUE)
        {
          (void)memcpy((VP)&msg.tm_msg,&pkt[1],pkt[1]);
          /* update size as per timestamp format */
          size += (sizeof(msg.tm_msg)- pkt[1]);
          pkt += (pkt[1]+ 1);
          buf = (VP)&msg.msg_ptn_1;
        }
        (void)memcpy((VP)&msg.trc_msg, &pkt[1],pkt[1]);

        /* Write trace packet */
        (void)lseek(fd, sz, SEEK_SET);
        sz = write(fd, buf, size);
        if((size_t)sz == size)
        {
          mgr->sh->prxy.cur_sz += size;
          TRACE_pack_32bit_in_bigendian(mgr->sh->prxy.cur_sz,
                                   (VP)&hdr.over_wr_cnt);
          rc = TRUE;
        }
      }
      else /* file size exceeds */
      {
        /* spl handling for case - can't overwrite exactly the complete file
         * since the current msg does not fit into the file there could be
         * one or more last messages which are not from current session
         */
        if(hdr.ms.mode_stat.file_stat == TRACE_BIN_FILE_PARTLY_FULL)
        {
          sz =  TRACE_get_file_sz(fd);
          sz -= (ssize_t)mgr->sh->prxy.cur_sz;
          if(sz > 0)
          {
            (void)lseek(fd, (S32)mgr->sh->prxy.cur_sz, SEEK_SET);
            written_bytes = write(fd, (VP)dummy_buf, (size_t)sz);
            if(0 == written_bytes)
            {
              TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Bytes written is Zero\n");
            }
          }
        }
        hdr.ms.mode_stat.file_stat = TRACE_BIN_FILE_FULL;
        mgr->sh->prxy.cur_sz = 0;
        mgr->sh->prxy.cur_bfile_cnt++;
        if(mgr->sh->prxy.cur_bfile_cnt > mgr->sh->prxy.max_bfile_cnt)
        {
          /* reset to 1 */
          mgr->sh->prxy.cur_bfile_cnt = 1;
        }
      }
      if((hdr.ms.mode_stat.file_stat == TRACE_BIN_FILE_PARTLY_FULL)||
        (hdr.ms.mode_stat.file_stat == TRACE_BIN_FILE_FULL))
      {
        /* update file stat */
        (void)lseek(fd,0, SEEK_SET);
         sz = write(fd, (VP)&hdr, sizeof(hdr));
         if((size_t)sz != sizeof(hdr))
         {
            break;
         }
      }
     (void)fsync(fd);
     (void)close(fd);
     fd = -1;
    }
  }
  if((E_FAIL == rc) && (fd >= E_OK))
  {
    (void)fsync(fd);
    (void)close(fd);
  }
  return rc;
}

EXPORT ER TRACE_bin_wr(TRACE_mgr* mgr, TRACE_q* curq)
{
  ER  rc    = E_OK;
  U32  wrsz  = 1;
  U8  rawpkt[sizeof(TRACE_TRACEOUT_MSG) + 2];

  TRACE_wr_common(mgr, curq, rawpkt, (VP)&wrsz);

  if(mgr->sh->prxy.flush_mode == TRUE)
  {
    rc = TRACE_flush_wr(mgr, rawpkt, wrsz);
  }
  else
  {
    rc = TRACE_ovr_wr(mgr, rawpkt, wrsz);
  }
  return rc;
}



/**
 * Save Trace proxy configuration onto external storage media.
 *
 * \parm mgr Pointer to Trace manager
 * \parm fd  File descriptor
 *
 * return None
 */
LOCAL ER TRACE_wr_proxy_cfg_to_storage_media(TRACE_mgr* mgr, S32 fd);
LOCAL ER TRACE_wr_proxy_cfg_to_storage_media(TRACE_mgr* mgr, S32 fd)
{
  size_t wrsz              = 0;
  U32    bfile_sz          = 0;
  U32    maxlog_sz         = 0;
  U16    clas_cnt           = 0;
  VP     buf               = NULL;
  U16     prev_clas_cnt     = 0;
  U16     new_clas_cnt      = 0;
  S32      stage             = 0;
  ER     rc                = E_OK;
  ssize_t asz              = 0;
  ssize_t sz               = 0;
  U8     blkmode	   = 0;
  U8     len               = mgr->trc.in_msg->cmd_pkt.hdr.len;
  TRACE_PROXY_CFG_HDR_1* hdr_1 =
  (TRACE_PROXY_CFG_HDR_1*)(VP)mgr->trc.in_msg->cmd_pkt.payld.payld;
  TRACE_PROXY_CFG_HDR_2* hdr_2 =
  (TRACE_PROXY_CFG_HDR_2*)(VP)(&hdr_1->file_nm_sz + hdr_1->file_nm_sz +
                               sizeof(hdr_1->file_nm_sz));

  while((stage < 8) && (rc >= E_OK))
  {
    switch(stage)
    {
     case 0: /* filenm length & filenm */
        if(FALSE == hdr_1->append)
        {
          buf  = &hdr_1->file_nm_sz;
          wrsz = sizeof(hdr_1->file_nm_sz)+ hdr_1->file_nm_sz;
        }
        break;

     case 1: /* calculate class count */
        clas_cnt = len - (sizeof(TRACE_IN_HDR)+
                          sizeof(hdr_1)       +
                          hdr_1->file_nm_sz   +
                          sizeof(hdr_2->timestamp_mode) +
                          sizeof(hdr_2->per_file_sz));
        clas_cnt = clas_cnt / (sizeof(TRACE_proxy_file_cl));
        if(FALSE == hdr_1->append)
        {
          buf  = &clas_cnt;
          wrsz = sizeof(clas_cnt);
        }
        break;

      case 2: /* Update class count */
        if(hdr_1->append == TRUE)
        {
          sz = (ssize_t)lseek(fd, (ssize_t)sizeof(hdr_1->file_nm_sz) + hdr_1->file_nm_sz,
                                                             SEEK_SET);
          if(sz < E_OK)
          {
            rc = E_FAIL;
          }
          else
          {
            sz = read(fd, (VP)&prev_clas_cnt, sizeof(prev_clas_cnt));
            if((size_t)sz == sizeof(prev_clas_cnt))
            {
              new_clas_cnt = clas_cnt + prev_clas_cnt;
              buf  = &new_clas_cnt;
              wrsz = sizeof(new_clas_cnt);
              sz = (ssize_t)lseek(fd, (S32)sizeof(hdr_1->file_nm_sz) + hdr_1->file_nm_sz,
                                                             SEEK_SET);
              if(sz < E_OK)
              {
                rc = E_FAIL;
              }
            }
          }
        }
        break;

      case 3: /* Class level info */
        if(hdr_1->append == TRUE)
        {
          /* per file sz is written as UW in the cfg file */
          sz = (ssize_t)lseek(fd, (sizeof(U32)+ sizeof(hdr_2->timestamp_mode)),
                                                         SEEK_END);
          if(sz < E_OK)
          {
            rc = E_FAIL;
          }
        }
        if(rc >= E_OK)
        {
          buf  = &hdr_2->clas_inf;
          wrsz = clas_cnt * (sizeof(TRACE_proxy_file_cl));
        }
        break;

      case 4: /* timestamp info */
        /* timestamp from TTFis msg is not used for bkward compatibility
         * use the one as updated by TR_ENABLE_PROXY_TIMESTAMP cmd
         */
        buf  = &mgr->sh->prxy.timestamp_flg;
        wrsz = sizeof(mgr->sh->prxy.timestamp_flg);
        break;

      case 5: /* per file sz */
        /* per file sz from TTFis cmd is not used for bkward compatibility
         * use the one as specified by Devconf.
         * This value is not used when prxy is initialized
         */
       // bfile_sz = mgr->devconf->prxy_log_sz;
        bfile_sz = mgr->prxy.filesz;
        buf = &bfile_sz;
        wrsz = sizeof(bfile_sz);
        break;
    
     case 6: /* max log size */
        maxlog_sz = mgr->prxy.max_logsz;
        buf = &maxlog_sz;
        wrsz = sizeof(maxlog_sz);
        break;	
		
      case 7: /* blockmode info */
        blkmode = mgr->prxy.blk_mode;
        buf = &blkmode;
        wrsz = sizeof(blkmode);
        break;

      default:
        break;
    }
    if((E_OK >= rc) && (wrsz > 0) && (buf != NULL))
    {
      asz = write(fd, buf, wrsz);
      if(asz != (ssize_t)wrsz)
      {
        rc = E_FAIL;
      }
      wrsz = 0;
    }
    stage++;
  }

  return rc;
}

/**
 * Save user provided configuration for Trace Proxy.
 *
 * \parm mgr Pointer to Trace manager
 *
 * return E_OK if successful
 *        \li  Error returned by concrete implementation.
 */
EXPORT ER TRACE_cre_proxy_cfg(TRACE_mgr* mgr)
{
  S32  fd                           = -1;
  ER     rc                         = E_OK;
  S32    mode                       = TRACE_FILE_W_PLUS;
  S8* file_nm                       = mgr->prxy.file_nm;
  TRACE_PROXY_CFG_HDR_1* hdr        = (TRACE_PROXY_CFG_HDR_1*)(VP)mgr->trc.in_msg->cmd_pkt.payld.payld;
    VP id = NULL;
    U8 loc[TRACE_DEVNM_LEN] = {0};
    U8 fullPath[TRACE_DEVNM_LEN+50] = {0};

    id = PFCFG_opn_cnf("adit_trace", PFCFG_READ);
    if(id)
    {
	  TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Opened PFCFG cfg file successfully, from proxy\n");
	  PFCFG_get_cfs(id, "TRACE_PRXY_CFG_LOC", (VP)&loc, TRACE_DEVNM_LEN);
    }
    else
    {
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Opening PFCFG cfg file FAILED from proxy \n");
    }

  if(TRUE == hdr->append)
  {
    mode = TRACE_FILE_R_PLUS; 
  }

  strcpy((VP)file_nm, (VP)mgr->sh->prxy.cfg_file_path);
  strcat((VP)file_nm, (VP)TRACE_PROXY_CFG_FILE_NM);
    sprintf((char*)fullPath, "mount -o remount rw %s", (char*)loc);
    if(E_FAIL == system((char*)fullPath))
    {
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Error in changing permission to read write\n");
    }

    fd = open((VP)(file_nm), mode, TRACE_FILE_ACCESS_MODE);

  if(fd >= E_OK)
  {
    rc = TRACE_wr_proxy_cfg_to_storage_media(mgr, fd);
    (void)fsync(fd);
    (void)close(fd);
  }
  else
  {
    TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Error in saving Proxy Configuration file\n");
    rc = E_FAIL;
  }
  sprintf((char*)fullPath, "mount -r -o remount %s", (char*)loc);
  if(E_FAIL == system((char*)fullPath))
  {
    TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Error in changing permission to read only\n");
  }
  return rc;
}

/**
 * Initialize Trace proxy.
 *
 * \param  mgr  Pointer to Trace manager
 *
 * \return E_OK if successful
 * \li     E_FAIL On occurence of error
 */
EXPORT ER TRACE_proxy_ini(TRACE_mgr* mgr)
{
  U16                  clas_cnt = 0;
  U8                   filenm_sz= 0;
  S32                  stage    = 0;
  U32                  i        = 0;
  S32                  fd       = -1;
  TRACE_proxy_file_cl* p        = NULL;
  U16    clas                   = 0;
  U8     compid                 = 0;
  U8     clasid                 = 0;
  ER     rc                     = E_OK;
  ssize_t asz                   = 0;
  S8* file_nm                   = mgr->prxy.file_nm;
  VP buf                        = NULL;
  size_t rdsz                   = 0;


  /* Open Trace Proxy configuration file */
  strcpy((VP)file_nm, (VP)mgr->sh->prxy.cfg_file_path);
  strcat((VP)file_nm, (VP)TRACE_PROXY_CFG_FILE_NM);
  fd = open((VP)file_nm, TRACE_FILE_RD_ONLY, TRACE_FILE_ACCESS_MODE);

  if(fd < E_OK)
  {
    rc = E_FAIL;
  }
  else
  {
    TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Proxy configuration file present\n");/*agv2kor*/
  }

  /* Read contents from Trace Proxy configuration file */
  while((stage < 8) && (rc >= E_OK))
  {
    switch(stage)
    {
      case 0: /* rd filenm sz */
        buf  = &filenm_sz;
        rdsz = sizeof(U8);
        break;

      case 1: /* log filenm */
        /* not used for backward compatibility */
        buf  = file_nm;
        rdsz = filenm_sz;
        file_nm[filenm_sz]= '\0';
        break;

      case 2: /* class cnt */
        buf  = &clas_cnt;
        rdsz = sizeof(clas_cnt);
        break;

      case 3: /* class lvl info */
        buf  = mgr->prxy.cl;/*As per the discussion no change required here*/
        rdsz = sizeof(TRACE_proxy_file_cl) * clas_cnt;
        break;

      case 4: /* timestamp info */
        buf  = &mgr->sh->prxy.timestamp_mode;
        rdsz = sizeof(mgr->sh->prxy.timestamp_mode);
        break;

      /* per file sz is read from Devconf instead of cfg
       * Otherwise, there could be inconsistency btw perfile sz and
       * max trace sz - read both values from current devconf
       */

      case 5: /* max file size */
        buf  = &mgr->prxy.filesz;
        rdsz = sizeof(mgr->prxy.filesz);
        break;

      case 6: /* max proxy log size */
        buf  = &mgr->prxy.max_logsz;
        rdsz = sizeof(mgr->prxy.max_logsz);
        break;
		
      case 7: /* blockmode info*/
        buf  = &mgr->prxy.blk_mode;
        rdsz = sizeof(mgr->prxy.blk_mode);
        break;
			  
      default:
        break;
    }
    if((fd >= E_OK) && (rdsz > 0) && (buf != NULL))
    {
      asz = read(fd, buf, rdsz);
      if((asz != (ssize_t)rdsz) && (stage < 5))
      {
        rc = E_FAIL;
      }
    }
    stage++;
  }

  if(rc >= E_OK)
  {
    if((mgr->sh->prxy.bfile_sz != 0) && (mgr->sh->prxy.stor_med_sz != 0))
    {

     /* If max file size or file size is changed by user, stored it to shmem */
      if((mgr->prxy.max_logsz != mgr->sh->prxy.stor_med_sz) || (mgr->prxy.filesz != mgr->sh->prxy.bfile_sz))
      {
        mgr->sh->prxy.bfile_sz    = mgr->prxy.filesz;
        mgr->sh->prxy.stor_med_sz = mgr->prxy.max_logsz;
      }
	  
      mgr->sh->prxy.bfile_sz = (mgr->sh->prxy.bfile_sz > mgr->sh->prxy.stor_med_sz)?
                            mgr->sh->prxy.stor_med_sz : mgr->sh->prxy.bfile_sz;
      mgr->sh->prxy.max_bfile_cnt = (mgr->sh->prxy.stor_med_sz)/(mgr->sh->prxy.bfile_sz);

      /* Initialize Trace class level info for Trace Proxy */
      for(i = 0, p = (TRACE_proxy_file_cl*)(VP)mgr->prxy.cl;
          i < clas_cnt; p++, i ++)
      {
        TRACE_get_16bit_frm_bigendian((VP)&p->clas,&clas);
        compid = (U8)TR_COMPID(clas);
        clasid = (U8)TR_CLASID(clas);
        if(clasid <= mgr->c_cnfg->comp[compid].max_clas)
        {
#ifdef TRACE_DLT_EXIST
          /*If trace_dlt_be proxy task is running, then update the trace level in trace buffer*/
          mgr->c_cnfg->comp[compid].lvl[clasid].trc_lvl = p->lvl;

          
#else          
        /* If trace_be proxy task is running, then update the proxy level in proxy buffer*/  
        mgr->c_cnfg->comp[compid].lvl[clasid].pxy_lvl = p->lvl;
        mgr->c_cnfg->comp[compid].lvl[clasid].trc_lvl = TR_ENABLED_CLASS_NONE ;
#endif
        }
      }
    }
    else
    {
      rc = E_FAIL;
    }
  }

  if(fd >= E_OK)
  {
    (void)close(fd);
  }
  return rc;
}

LOCAL void TRACE_prxy_cleanup(void* handler);
LOCAL void TRACE_prxy_cleanup(void* handler)
{
    TRACE_mgr* mgr = (TRACE_mgr*)handler;
    if(mgr != NULL)
    {
      /* unlocking cond mutex is required explicitly
       * especially when more than one thread waiting
       * on the same cond variable
       */
      mgr->sh->prxy.status = FALSE;
/* PRQA: Lint Message 455: unlocking cond mutex is required explicitly
   PRQA: Lint Message 455: especially when more than one thread waiting on the same cond variable */
/*lint -save -e455 */
     pthread_mutex_unlock(&mgr->sh->evt_id[EVT_PRXY_Q].EventMutex);
/*lint -restore */
    }
}


/**
 * Trace Proxy Task
 *
 * Initialize/Uninitilize Trace Proxy on external storage media
 * availability/unavailability.
 *
 * \param  stacd   Unused
 *         exinf   Pointer to trace manager
 *
 * \return None
 */
EXPORT VP TRACE_proxy_task(VP exinf)
{
  TRACE_mgr* mgr          = (TRACE_mgr*) exinf;
  ER         rc           = E_OK;
  S32        media_attach = FALSE;
  S32        media_detach = FALSE;
  volatile S32 stage      = 0;
  TRACE_q_status prxy_q   = TRACE_Q_PKTS_PENDING;
  DIR        *proxyPath;

  prctl(PR_SET_NAME, "TRACE_Proxy", 0, 0, 0);
  TRACE_flagAttr Attr     = {TRACE_PROXY_MEDIA_EVT, TWF_ORW | TWF_BITCLR,
                             0, TMO_FEVR};

    VP id = NULL;
    id = PFCFG_opn_cnf("adit_trace", PFCFG_READ);
    if(id)
    {
      TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Opened PFCFG cfg file successfully\n");
    }
    else
    {
      TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Opening PFCFG cfg file FAILED\n");
    }

    PFCFG_get_cfs(id, (char*)TRACE_PRXY_CFG_LOC, (char*)mgr->sh->prxy.cfg_file_path, TRACE_DEVNM_LEN);
    PFCFG_close_cnf(id);
    proxyPath = opendir((char*)mgr->sh->prxy.cfg_file_path);
    if(NULL!= proxyPath)
    {
       TRACE_set_flag(&mgr->sh->evt_id[EVT_PRXY_Q], (U32)TRACE_PROXY_MEDIA_ATTACH);
       TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Media attach signal from proxy task\n");
       closedir(proxyPath);
    }
  pthread_cleanup_push(TRACE_prxy_cleanup, mgr);

  /* PRQA: QAC Message 3201: All the statements get executed based on stage increment */
  /* PRQA S 3201 L2 */
  while((stage < 3) && (mgr->prxy.exit == FALSE) && (rc >= E_OK))
  {
    /* test if we are supposed to cancel */
    pthread_testcancel();

    switch(stage)
    {
      case 0:
           /*.bin file writing should be happen only from trace_be */
#ifndef TRACE_DLT_EXIST
    	     /* check for data availability */
            if((TRUE == mgr->prxy.media_attach) &&
               ((Attr.flg_ptn & TRACE_DAT_AVL_PRXY_BUF)== TRACE_DAT_AVL_PRXY_BUF))
            {
                  /* Get messages from queue */
               do
               {
           	 prxy_q = TRACE_q_pop(&mgr->prxy.buf);
           	 if(TRACE_Q_PKTS_PENDING == prxy_q)
            	 {
             	       /* send overwrite info */
              	     if(mgr->sh->prxy.buf.ovrwr_cnt_alias > 0)
                     {
                    	 (void)TRACE_snd_ovrcnt_info(mgr,&TRACE_bin_wr,
                         mgr->sh->prxy.buf.ovrwr_cnt_alias);
                         mgr->sh->prxy.buf.ovrwr_cnt_alias = 0;
                     }
                      /* write data to proxy */
                     rc = TRACE_bin_wr(mgr, &mgr->prxy.buf);
                 }
               }while((TRACE_Q_EMPTY != prxy_q) && (rc >= E_OK) &&
                                    (mgr->prxy.exit != TRUE));
               if(rc < E_OK)
               {  
                 /* Error will be returned when media is removed
                 * do not exit, instead wait for media attach evt
                 */
                   rc = E_OK;
                   TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Proxy bin writing failed\n");/*agv2kor*/
              }
          } 
#else
        /*to avoid the complier warning */
          prxy_q = prxy_q;

#endif

        break;

      case 1:
          rc = TRACE_wait_flag(&mgr->sh->evt_id[EVT_PRXY_Q], &Attr);
        break;
      case 2:
        /* process proxy events */
	TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "proxy: path of proxy file is = %s \n",mgr->sh->prxy.cfg_file_path);
        media_attach = ((Attr.flg_ptn & TRACE_PROXY_MEDIA_ATTACH) == TRACE_PROXY_MEDIA_ATTACH)? TRUE : FALSE;
        if(media_attach == TRUE )
        {
          TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Proxy Media attach event received\n");/*agv2kor*/
          if(FALSE == mgr->prxy.media_attach)
          {
          
		  char errMsg[TRACE_ERR_MEM_BLK_MODE_STR_SIZE]={0};
            rc = TRACE_proxy_ini(mgr);
          /*Set buffer mode*/
            if(rc >= E_OK)
            {
            /*SWGIII-6993-Fix:Block mode status updation in proxy in trace-dlt mode */
             #ifdef TRACE_DLT_EXIST
            	mgr->sh->trc.buf.blck_mode = (TRUE == mgr->prxy.blk_mode)? TRACE_NO_BLOCK_MODE : TRACE_BLOCK_MODE;
            	mgr->sh->prxy.buf.blck_mode = (TRUE == mgr->prxy.blk_mode)? TRACE_NO_BLOCK_MODE : TRACE_BLOCK_MODE;
            	mgr->sh->trc.d_cnfg.flgs.timestamp =(TRACE_TIMESTAMP_MODE)mgr->sh->prxy.timestamp_mode;

             #else
            	mgr->sh->prxy.buf.blck_mode = (TRUE == mgr->prxy.blk_mode)? TRACE_NO_BLOCK_MODE : TRACE_BLOCK_MODE;
             #endif
			 
			 sprintf( errMsg,"[proxy_attach] BLOCKMODE { TRACE: %d  PROXY : %d }",(TRACE_BLOCK_MODE == mgr->sh->trc.buf.blck_mode)? TRUE : FALSE,(TRACE_BLOCK_MODE == mgr->sh->prxy.buf.blck_mode)? TRUE : FALSE);
			 TRACE_errmem_s32IOWrite(errMsg, strlen(errMsg));
            }
            mgr->sh->prxy.lookup_filecnt = TRUE;
          }
#ifndef TRACE_DLT_EXIST /* trace_dlt_be shall not wait for proxy data, only trace_be waits for proxy data available signal */
          Attr.waiptn |= TRACE_PROXY_WAI_PTN;
#endif
        }
        else
        {
          media_detach = ((Attr.flg_ptn & TRACE_PROXY_MEDIA_DETACH) == TRACE_PROXY_MEDIA_DETACH)? TRUE : FALSE;
          if(media_detach == TRUE)
          {
          char errMsg[TRACE_ERR_MEM_BLK_MODE_STR_SIZE]={0};
         /*When media detach time set the buffer mode as non block mode*/
        	  /*SWGIII-6993-Fix:Block mode status updation in proxy in trace-dlt mode */
            #ifndef TRACE_DLT_EXIST
        	if(TRACE_BLOCK_MODE == mgr->sh->prxy.buf.blck_mode)
            {
              mgr->sh->prxy.buf.blck_mode = TRACE_NO_BLOCK_MODE;
              TRACE_q_release_wait_tasks(&mgr->prxy.buf);
            }
            #else
        	if((TRACE_BLOCK_MODE == mgr->sh->trc.buf.blck_mode)&&(TRACE_BLOCK_MODE == mgr->sh->prxy.buf.blck_mode))
        	{
        	   mgr->sh->trc.buf.blck_mode = TRACE_NO_BLOCK_MODE;
        	   /*This is needed for  updation of No-Block mode status in configuration file */
        	   mgr->sh->prxy.buf.blck_mode = TRACE_NO_BLOCK_MODE;
        	   TRACE_q_release_wait_tasks(&mgr->trc.buf);
        	}
        	 mgr->sh->trc.d_cnfg.flgs.timestamp = TRACE_TIMESTAMP_DISABLE;
           #endif
			
		   sprintf( errMsg,"[proxy_detach] BLOCKMODE { TRACE: %d  PROXY : %d }",(TRACE_BLOCK_MODE == mgr->sh->trc.buf.blck_mode)? TRUE : FALSE,(TRACE_BLOCK_MODE == mgr->sh->prxy.buf.blck_mode)? TRUE : FALSE);
		   TRACE_errmem_s32IOWrite(errMsg, strlen(errMsg));
            TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Proxy Media detach event received\n");/*agv2kor*/
            /* go and wait for media attach evt */
            stage = 0;
            Attr.waiptn = TRACE_PROXY_MEDIA_EVT;
          }
        }
        if((rc >= E_OK) && ((media_attach == TRUE) || (media_detach == TRUE)) )
        {
          mgr->prxy.media_attach = (media_attach == TRUE)? TRUE : FALSE;
          mgr->sh->prxy.status = mgr->prxy.media_attach;
	  /*status used for updating for TR_GET_TRACE_SETTINGS command from TTFis  */
          mgr->sh->prxy.config_status = mgr->prxy.media_attach;
          media_attach = FALSE;
          media_detach = FALSE;

        }
	else if (rc != E_OK)
	{
	  rc = E_OK; /*to avoid killing of proxy task*/
	  /*SWGIIX-2283 (SWGIIX-2142) [b_MAIN] Trace Proxy doesn't write to USB memory on ADIT environment.*/
	}
	else
	{
	  /*do nothing*/
	}
        break;

      default:
        break;
    }
    stage++;
    /* loop until proxy exits or error occurs */
    stage = (stage == 3)? 0 : stage;
  }
    /* PRQA L:L2 */
  pthread_cleanup_pop(1);
  return NULL;
}

